﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;
using Microsoft.Xna.Framework.Content;

namespace WinFormsGraphicsDevice
{
    public class Engine
    {
        private GameServiceContainer _services;
        private GraphicsDevice _graphicsDevice;
        private ContentManager _content;
        private SpriteBatch _spriteBatch;

        private Texture2D _xor;
        private Vector2 _xorPosition;
        private Vector2 _xorDirection;

        public Engine (IGraphicsDeviceService graphics)
        {
            _services = new GameServiceContainer();
            _services.AddService(typeof(IGraphicsDeviceService), graphics);

            _graphicsDevice = graphics.GraphicsDevice;

            _content = new ContentManager(_services);
            _spriteBatch = new SpriteBatch(_graphicsDevice);

            ClearColor = Color.DarkOrchid;
        }

        public void Initialize ()
        {
            _xor = BuildXorTexture(_graphicsDevice, 6);
            _xorDirection = new Vector2(.5f, .5f);
        }

        public void Update (GameTime gameTime)
        {
            float limitX = _graphicsDevice.Viewport.Width - _xor.Width;
            float limitY = _graphicsDevice.Viewport.Height - _xor.Height;

            if (_xorPosition.X >= limitX && _xorDirection.X > 0)
                _xorDirection.X *= -1;
            if (_xorPosition.X <= 0 && _xorDirection.X < 0)
                _xorDirection.X *= -1;
            if (_xorPosition.Y >= limitY && _xorDirection.Y > 0)
                _xorDirection.Y *= -1;
            if (_xorPosition.Y <= 0 && _xorDirection.Y < 0)
                _xorDirection.Y *= -1;

            _xorPosition += _xorDirection;
        }

        public void Draw (GameTime gameTime)
        {
            _graphicsDevice.Clear(ClearColor);

            _spriteBatch.Begin();
            _spriteBatch.Draw(_xor, _xorPosition, Color.White);
            _spriteBatch.End();
        }

        public Color ClearColor { get; set; }

        private static Texture2D BuildXorTexture (GraphicsDevice device, int bits)
        {
            Texture2D tex = new Texture2D(device, 1 << bits, 1 << bits);
            Color[] data = new Color[tex.Width * tex.Height];

            for (int y = 0; y < tex.Height; y++) {
                for (int x = 0; x < tex.Width; x++) {
                    float lum = ((x << (8 - bits)) ^ (y << (8 - bits))) / 255f;
                    data[y * tex.Width + x] = new Color(lum, lum, lum);
                }
            }

            tex.SetData(data);
            return tex;
        }
    }
}
